/*
*********************************************************************************************************
*                                                ALLD_lib.cpp
*
* Description : This is the implementation of header file.
*
*
*
*********************************************************************************************************
*/


/*
************************************************************************************************************************
*                                                 INCLUDE LIBRARIES FILES
************************************************************************************************************************
*/

#include <thread>
#include <vector>
#include <ostream>
#include <opencv2/core.hpp>
#include <iostream>
#include <opencv2/opencv.hpp>
#include <stdio.h>
#include <stdarg.h>
#include <math.h>
#include "ALLD_lib.h"

/*
************************************************************************************************************************
*                                                 SET NAMESPACES
************************************************************************************************************************
*/

using namespace cv;
using namespace std;

// Use this as a trigger visualization
const bool verbose = false; // if you want to see the intermediate steps change it to true

/*
************************************************************************************************************************
*                                              linspace FUNCTION
*
* Description: This function is quite similar to the numpy.linspace python function.
			   It returns a vector which contain equidistant numbers (num) over a specified interval (start, stop)
*
*
************************************************************************************************************************
*/

template<typename T>
std::vector<double> linspace(T start_in, T end_in, int num_in)
{
	std::vector<double> linspaced;
	double start = static_cast<double>(start_in);
	double end = static_cast<double>(end_in);
	double num = static_cast<double>(num_in);

	if (num == 0) {
		return linspaced;
	}
	if (num == 1) {
		linspaced.push_back(start);
		return linspaced;
	}

	double delta;

	delta = (end - start) / (num - 1);

	for (int j = 0; j < num - 1; ++j) {
		linspaced.push_back(start + delta * j);
	}

	linspaced.push_back(end);

	return linspaced;
}

/*
************************************************************************************************************************
*                                              show_and_save FUNCTION
*
* Description: This function is used to display and possibly save an image on the screen preserving its extension
*
*
************************************************************************************************************************
*/

void show_and_save(std::string filename, cv::Mat image, std::string file_extension, bool show, bool save) {

	if (show == true) {
		namedWindow(filename, WINDOW_AUTOSIZE);  // set the image window
		imshow(filename, image);				 // show the image inside the window
		waitKey(0);								 // wait for any key press
		destroyWindow(filename);				 // close the window
	}

	if (save == true) {

		std::string path = "./out/" + filename + "." + file_extension;

		bool isSuccess = cv::imwrite(path, image); //write the image to a file

		if (isSuccess == false)
		{
			cout << "Error...Failed to save the image" << endl;
			cin.get(); //wait for a key press
			CV_Error(Error::StsError, "Unable to save image: " + filename);
			exit(EXIT_FAILURE);
		}
		cout << filename << " image is successfully saved to file: " << path << endl;
	}
	return;
}

/*
************************************************************************************************************************
*                                              getFileExt FUNCTION
*
* Description: This function is used to get the extension of the input file returning it as a string
*
*
************************************************************************************************************************
*/

std::string getFileExt(const std::string& s) {

	size_t i = s.rfind('.', s.length());
	if (i != string::npos) {
		return(s.substr(i + 1, s.length() - i));
	}

	return("");
}

/*
************************************************************************************************************************
*                                              print_vector FUNCTION
*
* Description: This function allows to print a vector with its size
*
*
************************************************************************************************************************
*/

void print_vector(std::vector<double> vec)
{
	std::cout << "Vector size: " << vec.size() << std::endl;
	for (double j : vec)
		std::cout << j << " ";
	std::cout << std::endl;

	return;
}

/*
************************************************************************************************************************
*                                              get_histogram FUNCTION
*
* Description: This function is used to get the initial positions of lane lines by using the lower half of the image
			   to calculate the histogram used later to detect 2 peaks that represent the correct position of each lane line
*
*
************************************************************************************************************************
*/

void get_histogram(cv::Mat const& input, cv::Mat& hist)
{
	// Reduce Image (Size(1280, 720) to 1 x 720 and sum the row elements
	cv::Mat bottom_half_y;
	// Take a histogram of the bottom half of the image
	bottom_half_y = input.rowRange(input.size().height / 2, input.size().height);
	// Reduces the matrix to a vector summing the all the row elements obtaining the histogram
	reduce(bottom_half_y, hist, 0, REDUCE_SUM, CV_32FC1);

	return;
}

/*
************************************************************************************************************************
*                                              draw_poly_lines FUNCTION
*
* Description: This function is able to retrieve the right and left line pixel positions used to compute a second degree polynomial,
			   through polyfit() function, in order to find the coefficients of the curves that best fit the left and right lane lines.
			   After that a vector of Points is used to store the x and y positions of these best fitted lines drawing them in the output
			   image with the window lines +/- margin (skip sliding window).
			   Finally, generate a polygon to illustrate the search window area for left and right lines
*
*
************************************************************************************************************************
*/

Points draw_poly_lines(Mat const& input, vector<Point>& good_left_inds, vector<Point>& good_right_inds, vector<Point>& leftFitPt, vector<Point>& rightFitPt, vector<float>& leftFit_x, vector<float>& rightFit_x)
{
	Points pt;
	int height = input.size().height;  // image height
	
	// Extract left and right line pixel positions and store their x and y coordinates into a vector
	std::vector<float> leftx, lefty, rightx, righty;

	for (Point good_left : good_left_inds) {
		leftx.push_back(float(good_left.x));
		lefty.push_back(float(good_left.y));
	}

	for (Point good_right : good_right_inds) {
		rightx.push_back(float(good_right.x));
		righty.push_back(float(good_right.y));
	}

	// Fit a second order polynomial to each line pixel positions

	const int order = 2;
	int left_Row, right_Row;
								
	left_Row = (int)leftx.size();
	right_Row = (int)rightx.size();

	// Prepare the data for the least squares polynomial fit
	Mat leftxMat(left_Row, 1, CV_32FC1, leftx.data());
	Mat leftyMat(left_Row, 1, CV_32FC1, lefty.data());
	Mat leftDst(3, 1, CV_32FC1);

	// Find the coefficients of polynomial equation for left and right line
	polyfit(leftyMat, leftxMat, leftDst, order);

	Mat rightxMat(right_Row, 1, CV_32FC1, rightx.data()); // Return a pointer to the first element in the rightx array
	Mat rightyMat(right_Row, 1, CV_32FC1, righty.data()); // Return a pointer to the first element in the righty array
	Mat rightDst(3, 1, CV_32FC1);

	polyfit(rightyMat, rightxMat, rightDst, order);

	// Take the left and right polynomial coefficients
	float left_coeff[3] = { leftDst.at<float>(2,0), leftDst.at<float>(1,0), leftDst.at<float>(0,0) };
	float right_coeff[3] = { rightDst.at<float>(2,0), rightDst.at<float>(1, 0), rightDst.at<float>(0,0) };
	float left_fitx, right_fitx;
	const int margin = 50;
	
	// Apply a polinomial fit of order 2: p(y) = coeff[0] * y^2 + coeff[1] * y + coeff[2]
	// Calculate x pixel position based on input image height and coefficients
	// Set the area of search (skip sliding window) based on activated x values within the + / - margin of polynomial function
	for (int i = 0; i < height; ++i) {

		left_fitx = left_coeff[0] * (float)pow(i, order) + left_coeff[1] * i + left_coeff[2];
		leftFit_x.push_back(left_fitx);
		right_fitx = right_coeff[0] * (float)pow(i, order) + right_coeff[1] * i + right_coeff[2];
		rightFit_x.push_back(right_fitx);
		leftFitPt.push_back(Point(int(left_fitx), i));
		rightFitPt.push_back(Point(int(right_fitx), i));
		
		// Left Curve + / - margin
		pt.leftFit_windowLine1.push_back(Point(int(left_fitx) - margin, i)); 
		pt.leftFit_windowLine2.push_back(Point(int(left_fitx) + margin, i)); 

		// Right Curve + / - margin
		pt.rightFit_windowLine1.push_back(Point(int(right_fitx) - margin, i)); 
		pt.rightFit_windowLine2.push_back(Point(int(right_fitx) + margin, i));
	}

	// Color used to draw the window lines
	cv::Scalar YELLOW = Scalar(0, 255, 255);
	cv::Scalar GREEN = Scalar(0, 255, 0);

	// Draw right and left best-fit lines
	cv::polylines(input, leftFitPt, false, YELLOW, 3);
	cv::polylines(input, rightFitPt, false, YELLOW, 3);

	// Draw right and left fit lines + / - margins
	cv::polylines(input, pt.leftFit_windowLine1, false, YELLOW);
	cv::polylines(input, pt.leftFit_windowLine2, false, YELLOW);
	cv::polylines(input, pt.rightFit_windowLine1, false, YELLOW);
	cv::polylines(input, pt.rightFit_windowLine2, false, YELLOW);

	// Set the left and right window lines
	cv::Mat out_img;
	Mat windowCurve = Mat::zeros(input.size(), input.type());
	vector<Point> left = pt.leftFit_windowLine1;
	vector<Point> right = pt.rightFit_windowLine1;

	reverse(pt.leftFit_windowLine2.begin(), pt.leftFit_windowLine2.end());
	reverse(pt.rightFit_windowLine2.begin(), pt.rightFit_windowLine2.end());
	left.insert(left.end(), pt.leftFit_windowLine2.begin(), pt.leftFit_windowLine2.end());
	right.insert(right.end(), pt.rightFit_windowLine2.begin(), pt.rightFit_windowLine2.end());
	
	// Fill the polygon with the window lines
	cv::fillConvexPoly(windowCurve, left, GREEN);
	cv::fillConvexPoly(windowCurve, right, GREEN);
	
	// Blending the images into a single output image
	cv::addWeighted(input, 1, windowCurve, 0.3, 0, out_img);

	// Show the final result
	// 	cv::imshow("Sliding Windows", out_img);
	// 	cv::waitKey(0);
	// 	cv::destroyWindow("Sliding Windows");

	return pt;
}

/*
************************************************************************************************************************
*                                              sliding_window FUNCTION
*
* Description: This function is used to get the pixel coordinates so that to identify and follow lane lines pixels.
			   To do so, the following steps are applied:
			   1) In the thresholded binary warped image, pixel values are either 0 or 1, so using the histogram, the pixel values along each column in the image are summed
				  in order to find the most significant peaks in the histogram that represent the starting points of the x-position of the lane lines
			   2) Set the Hyper-parameters of the windows
			   3) From that starting points, a sliding window placed in the centers of the lines is used to find and follow the lines up to the top of the image.
				  If a line pixel in the window is identified re-centering the next windows correctly based on the mean position
			   4) Fit a second order polynomial to each line pixel using draw_poly_lines() function and displaying them
*
*
************************************************************************************************************************
*/

Points sliding_window(Mat const& input, Mat& hist, Mat& win, vector<Point>& good_left_inds, vector<Point>& good_right_inds, vector<Point>& leftFitPt, vector<Point>& rightFitPt, vector<float>& leftFit_x, vector<float>& rightFit_x)
{
	// struct of vector points which will contain window area for left and right lines
	Points pt; 
	// Find the peak of the left and right halves of the histogram, these will be the starting point for the left and right lines
	int midPoint = (int)hist.cols / 2;							// width / 2
	cv::Point leftx_base, rightx_base;							// starting point of left and right lines
	cv::Mat left_half = hist.colRange(0, midPoint);				// left part of the histogram
	cv::Mat right_half = hist.colRange(midPoint, hist.cols);	// right part of the histogram

	//Find out the peak (maximum value) for left and right lines
	cv::minMaxLoc(left_half, nullptr, nullptr, nullptr, &leftx_base);
	cv::minMaxLoc(right_half, nullptr, nullptr, nullptr, &rightx_base);
	rightx_base.x += midPoint;

	// Left and right current positions to be updated
	cv::Point leftx_current = leftx_base;
	cv::Point rightx_current = rightx_base;

	// Mean position of the left and right line pixels
	cv::Scalar left_mean, right_mean;

	// Sum 1d channel image and make 3d channel image
	cv::Mat tmp[3] = { input, input, input };
	cv::Mat out_img;
	// Creates an output multichannel array out of several single-channel ones used to draw on and visualize the result
	cv::merge(tmp, 3, out_img);

	// set the appearance of the windows (window settings: Hyper-parameters)
	// Choose the number of sliding windows
	const int nwindows = 12;
	// Set the width of the windows +/- margin
	const int margin = 50;
	//Set the minimum number of pixels found to recenter window
	const int minpix = 50;
	// Set the color of the windows
	cv::Scalar GREEN = cv::Scalar(0, 255, 0);
	// Set the height of windows based on nwindows and image shape
	int windowHeight = (int)input.size().height / nwindows;

	// Identify the x and y positions of all nonzero (i.e.activated) pixels in the image
	std::vector<cv::Point> nonzero;
	// Find out the position of nonzero pixels (i.e. activated) and store it to "nonzero" vector of points
	findNonZero(input, nonzero);

	int flag_L = 0;
	int flag_R = 0;

	// Iterate through each window in nwindows to track line pixels using for loop
	for (int window = 0; window < nwindows; ++window) {

		// Identify window boundaries in x and y (for right and left)
		int win_y_low = input.size().height - ((window + 1) * windowHeight);
		int win_y_high = input.size().height - (window * windowHeight);

		int win_xleft_low = leftx_current.x - margin;
		int win_xleft_high = leftx_current.x + margin;

		int win_xright_low = rightx_current.x - margin;
		int win_xright_high = rightx_current.x + margin;

		// Draw the windows on the visualization image
		cv::rectangle(out_img, Point(win_xleft_low, win_y_low), Point(win_xleft_high, win_y_high), GREEN, 2);
		cv::rectangle(out_img, Point(win_xright_low, win_y_low), Point(win_xright_high, win_y_high), GREEN, 2);

		// Index used to calculate mean points of the left and right line pixels for window box
		std::vector<cv::Point>left_lane_inds, right_lane_inds;

		// Identify the nonzero pixels in x and y within the window and store their indices in a vector of points (for left and right lines)
		for (Point non_zero : nonzero) {
			if ((non_zero.y >= win_y_low) && (non_zero.y < win_y_high) && (non_zero.x >= win_xleft_low)
				&& (non_zero.x < win_xleft_high)) {
				left_lane_inds.push_back(non_zero);			// Adds index elements to the bottom of the vector
				good_left_inds.push_back(non_zero);			//  best-fit pixels in windows of left lane
			}

			if ((non_zero.y >= win_y_low) && (non_zero.y < win_y_high) && (non_zero.x >= win_xright_low)
				&& (non_zero.x < win_xright_high)) {
				right_lane_inds.push_back(non_zero);		// Adds index elements to the bottom of the vector
				good_right_inds.push_back(non_zero);		// best-fit pixels in windows of right lane
			}
		}

		// Check the index vector size for left and right side
		// Check also if the lines are continuous or not
		if (left_lane_inds.size() == 0) {
			flag_L += 1;
			if (flag_L == 1) {
				cout << "Left lane line broken" << endl;
			}
			left_mean = Scalar(0, 0, 0, 0);
		}
		if (right_lane_inds.size() == 0) {
			flag_R += 1;
			if (flag_R == 1) {
				cout << "Right lane line broken" << endl;
			}
			right_mean = Scalar(0, 0, 0, 0);
		}

		// Calculate an average of the array elements (both for left and right)
		if (left_lane_inds.size() != 0) {
			left_mean = mean(left_lane_inds);
		}
		if (right_lane_inds.size() != 0) {
			right_mean = mean(right_lane_inds);
		}

		// If you found > minpix pixels and not black pixels, recenter next window (leftx_current or rightx_current) on their mean position
		// updating x point position of left and right
		if (left_mean != cv::Scalar(0, 0, 0, 0) && good_left_inds.size() > minpix)
			leftx_current = cv::Point2d(left_mean[0], left_mean[1]);

		if (right_mean != cv::Scalar(0, 0, 0, 0) && good_right_inds.size() > minpix)
			rightx_current = cv::Point2d(right_mean[0], right_mean[1]);

	}
	
	// check if the lines are correctly detected or not
	if (flag_L == nwindows) {
		cout << "Left lane line not correctly detected!" << endl;
	}

	if (flag_R == nwindows) {
		cout << "Right lane line not correctly detected!" << endl;
	}

	// 	cv::imshow("Sliding_Window", out_img);
	// 	cv::waitKey(0); 
	// 	cv::destroyWindow("Sliding_Window");

	// Fit a second order polynomial to each line pixel
	pt = draw_poly_lines(out_img, good_left_inds, good_right_inds, leftFitPt, rightFitPt, leftFit_x, rightFit_x);

	win = out_img.clone();

	return pt;
}

/*
************************************************************************************************************************
*                                              clear_Fit_Point FUNCTION
*
* Description: This function is used to clear the elements of a vector points, by removing all its elements (size = 0)
*
*
************************************************************************************************************************
*/


void clear_Fit_Point(vector<Point>& leftFitPt, vector<Point>& rightFitPt, Points& pt)
{
	leftFitPt.clear();
	rightFitPt.clear();

	pt.leftFit_windowLine1.clear();
	pt.leftFit_windowLine2.clear();

	pt.rightFit_windowLine1.clear();
	pt.rightFit_windowLine2.clear();

	return;
}

/*
************************************************************************************************************************
*                                              draw_lane_lines FUNCTION
*
* Description: This function is used to draw the lane lines on the final image.
			   After that, clear the elements of vector points with clearFitPoint_Vec function.
*
*
************************************************************************************************************************
*/

void draw_lane_lines(cv::Mat const& input, cv::Mat& finale, cv::Mat Minv, vector<Point>& leftFitPt, vector<Point>& rightFitPt, Points &pt)
{
	Mat Area = Mat::zeros(input.size(), input.type());

	// Set the color for polygon, left and right lane lines
	Scalar GREEN = Scalar(0, 255, 0);
	Scalar RED = Scalar(50, 0, 255);
	Scalar BLUE = Scalar(255, 0, 0);

	std::vector<cv::Point> poly_area = leftFitPt;
	reverse(rightFitPt.begin(), rightFitPt.end()); // Reverses the order of the elements in the range [first,last) to create the polygon's area
	poly_area.insert(poly_area.end(), rightFitPt.begin(), rightFitPt.end());

	// Fills the area bounded by the polygon and draw the lane lines
	fillConvexPoly(Area, poly_area, GREEN);
	polylines(Area, leftFitPt, false, RED, 20);
	polylines(Area, rightFitPt, false, BLUE, 20);

	// Retrieve the normal view using Minv by means of the inverse perspective transformation
	cv::Mat new_warp;
	cv::warpPerspective(Area, new_warp, Minv, Area.size(), cv::INTER_LINEAR); // get the normal view image 

	// Blending the images into a single output image
	cv::addWeighted(input, 1, new_warp, 0.3, 0, finale);

	return;
}

/*
************************************************************************************************************************
*                                              measure_curvature FUNCTION
*
* Description: This function is used to calculate the radius of curvature R of polynomial functions in meters
			   both for left and right lane lines.
*
*
************************************************************************************************************************
*/

float measure_curvature(vector<float>& fit_x, int height) {

	std::vector<double> fity = linspace<double>(0, ((double)height - 1), height);

	if (fity.size() != height) {
		cout << "Error..." << endl;
		exit(-1);
	}
	// Parameters used for the conversion in x and y from pixels space to meters (world space)
	int lane_px_height = 720;
	int lane_px_width = 700;

	double ym_per_pix = (30. / lane_px_height);		// meters per pixels in y dimension (lane length)
	double xm_per_pix = 3.5 / lane_px_width;		// meters per pixels in x dimension (lane width)
	
	int fit_Row = (int)fit_x.size();

	// Prepare the data for the least squares polynomial fit
	Mat fitxMat(fit_Row, 1, CV_32FC1, fit_x.data());

	// Choose the max y-value, correspond to the bottom of the image
	float y_eval = float(*(max_element(fity.begin(), fity.end())));

	cv::Mat y(fity);
	y.convertTo(y, CV_32FC1);

	cv::Mat poly_cr(3, 1, CV_32FC1);
	
	// Fit new polynomials to x, y in world space
	int order = 2;

	polyfit(y * ym_per_pix, fitxMat * xm_per_pix, poly_cr, order);
		
	// Retrieve the polynomial coefficients for left and right lane lines
	float coeff[3] = { poly_cr.at<float>(2,0), poly_cr.at<float>(1,0), poly_cr.at<float>(0,0) };

	// Calculation of the Radius of curvature both for the right and the left lane lines
	// f'(y) = dx/dy = 2Ay + B
	float derivative_1 = 2 * coeff[0] * y_eval * (float)ym_per_pix + coeff[1];
	// f''(y) = d^2x/dy^2 = 2A
	float derivative_2 = 2 * coeff[0];
	// R = ((1+(2Ay+B)^2)^3/2)/|2A|
	float curverad = pow((1 + pow(derivative_1, 2)), float(1.5)) / abs(derivative_2);

	return curverad;

}

/*
************************************************************************************************************************
*                                              measure_vehicle_offset FUNCTION
*
* Description: This function is used to calculate the offset of the vehicle and its direction in meters
			   with respect to the center of the lane.
			   Vehicle's offset calculation is based on 2 assumptions:
	           1) camera mounted in the front center of vehicle
		       2) road lane is 3.5 meters wide (from CARLA Simulator)
*
*
************************************************************************************************************************
*/

float measure_vehicle_offset(vector<float>& leftFit_x, vector<float>& rightFit_x, string& direction, int width) {

	float camera_center = 0;
	float lane_center = 0;

	if (!leftFit_x.empty() && !rightFit_x.empty()) {
		// Calculate the lane center
		lane_center = (rightFit_x.back() + leftFit_x.back())/2;
		// The horizontal center of the image
		camera_center = (float)width / 2;
	}

	float xm_per_pix = 3.5f / 700; //  meters per pixel in x dimension (lane width)
	float offset_pixels = (lane_center - camera_center);
	
	// Convert the vehicle's offset from pixels to meters
	float offset = offset_pixels * xm_per_pix;

	// Calculate the offset direction of the vehicle
	if (offset < 0) {
		direction = "left";
	}
	else {
		direction = "right";
	}

	if (verbose) {
		cout << "Direction: " << direction << endl;
		cout << "Deviation: " << offset << "m" << endl;
	}

	return offset;
}

/*
*********************************************************************************************************
*                                            change_Size FUNCTION
*
* Description : This function resizes an image/video frame to HD resolution (1280 x 720)
*
*
*********************************************************************************************************
*/

void change_Size(cv::Mat const& input, cv::Mat& dest, cv::Size size) {

	// Change the size of the image to HD definition -> Width: 1280 x Height: 720
	
	resize(input, dest, size);

	return;
}

/*
*********************************************************************************************************
*                                            Gauss_Blur FUNCTION
*
* Description : This function can be used to blur an image using a Gaussian filter
*
*********************************************************************************************************
*/

void Gauss_Blur(cv::Mat const& img, cv::Mat& dst, int kernel) {

	cv::GaussianBlur(img, dst, cv::Size(kernel, kernel), 0, 0, BORDER_DEFAULT);
	return;
}

/*
*********************************************************************************************************
*                                            abs_sobel_threshold FUNCTION
*
* Description : This function is used to get the thresholded Gradient in x and y direction using absolute value
				of Sobel operator to identify gradients (i.e. changing in color intensity in an image) from a gray scale image.
				To do so, the following steps are applied:

				1) Take the derivative from the gray scale image in x or y given orient = 'x' or 'y' using Sobel operator
				2) Calculates the absolute value of the derivative (Sobel_x or Sobel_y)
				3) Scales, and converts the result to 8-bit (0 - 255)
				4) Create a mask of 1's where the scaled gradient magnitude is >= thresh_min and <= thresh_max
				5) Return the gradient binary output image
*
*
*********************************************************************************************************
*/

void abs_sobel_threshold(cv::Mat const& src, cv::Mat& dest, char orient, int kernel_size, int thresh_min, int thresh_max)
{
	int dx, dy;
	// type of data of image
	int ddepth = CV_64F; //4 channel matrices

	cv::Mat grad, scale;

	if (orient == 'x') {
		dy = 0;
		dx = 1;
	}
	else {
		dy = 1;
		dx = 0;
	}

	cv::Sobel(src, grad, ddepth, dx, dy, kernel_size, 1, 0, BORDER_DEFAULT);
	grad = cv::abs(grad);

	// Scaling 
	double min, max;
	// Finds the global minimum and maximum in grad
	cv::minMaxLoc(grad, &min, &max);
	scale = 255 * (grad / max);
	// Rescale to 8-bit integers (0 - 255)
	scale.convertTo(scale, CV_8UC1); // type CV_8UC1 means a 8-bit single-channel array

	// Make sure that the conversion has been successful
	// This is the expression to be evaluated
	assert(scale.type() == CV_8UC1); // If this expression evaluates to 0, this causes an assertion failure that terminates the program

	// Checks that the range of "scale" lie between the gradient threshold_min and threshold_max
	cv::inRange(scale, cv::Scalar(thresh_min), cv::Scalar(thresh_max), dest);

	return;
}

/*
*********************************************************************************************************
*                                            magnitude_threshold FUNCTION
*
* Description : This function is used to get the Thresholded gradient Magnitude image from a gray scale image.
				To do so, the following steps are applied:

				1) Take the gradient in x and y separately from the gray scale image with Sobel operator
				2) Calculate the gradient magnitude
				3) Scales, and converts the result to 8-bit (0 - 255)
				4) Create a binary mask where gradient magnitude thresholds are met
				5) Return this mask as gradient magnitude binary output image
*
*********************************************************************************************************
*/

void magnitude_threshold(cv::Mat const& src, cv::Mat& dest, int kernel_size, int thresh_min, int thresh_max)
{
	int ddepth = CV_64F; //4 channel matrices

	cv::Mat magn_x, magn_y, abs_sobel, magnitude;

	// Apply Sobel operator to get the gradient in x and y
	cv::Sobel(src, magn_x, ddepth, 1, 0, kernel_size, 1, 0, BORDER_DEFAULT);
	cv::Sobel(src, magn_y, ddepth, 0, 1, kernel_size, 1, 0, BORDER_DEFAULT);

	// Calculates the magnitude
	cv::magnitude(magn_x, magn_y, abs_sobel);

	// Scaling 
	double min, max;
	// Finds the global minimum and maximum in an array
	cv::minMaxLoc(abs_sobel, &min, &max);
	magnitude = 255 * (abs_sobel / max);
	// Rescale to 8-bit integers (0 - 255)
	magnitude.convertTo(magnitude, CV_8UC1); // type CV_8UC1 means a 8-bit single-channel array

	// Make sure that the conversion has been successful
	// This is the expression to be evaluated
	assert(magnitude.type() == CV_8UC1); // If this expression evaluates to 0, this causes an assertion failure that terminates the program.

	// Checks that the range of "magnitude" lie between the magnitude threshold_min and threshold_max
	cv::inRange(magnitude, cv::Scalar(thresh_min), cv::Scalar(thresh_max), dest);

	return;
}

/*
*********************************************************************************************************
*                                            direction_threshold FUNCTION
*
* Description : This function is used to get the Thresholded gradient Direction image from a gray scale image
				To do so, the following steps are applied:

				1) Take the gradient in x and y separately from the gray scale image with Sobel operator
				2) Take the absolute value of the x and y gradients
				3) Calculate the gradient direction
				4) Create a binary mask where gradient direction thresholds are met
				5) Return this mask as gradient direction binary output image
*
*********************************************************************************************************
*/

void direction_threshold(cv::Mat const& src, cv::Mat& dest, int kernel_size, double thresh_min, double thresh_max)
{
	int ddepth = CV_64F; //4 channel matrices
	bool angleInDegrees = false; // to get the result in radians
	cv::Mat dir_x, dir_y;
	Mat orientation;

	// Apply Sobel operator to get the gradient in x and y
	cv::Sobel(src, dir_x, ddepth, 1, 0, kernel_size, 1, 0, BORDER_DEFAULT);
	cv::Sobel(src, dir_y, ddepth, 0, 1, kernel_size, 1, 0, BORDER_DEFAULT);

	// Takes their absolute values
	dir_x = cv::abs(dir_x);
	dir_y = cv::abs(dir_y);

	// Calculates the rotation angle
	cv::phase(dir_x, dir_y, orientation, angleInDegrees);

	// Checks that the range of "orientation" lie between the direction threshold_min and threshold_max
	cv::inRange(orientation, cv::Scalar(thresh_min), cv::Scalar(thresh_max), dest);

	return;
}

/*
*********************************************************************************************************
*                                            apply_threshold FUNCTION
*
* Description : This function is used to combine together different thresholded gradient binary images in
				one single binary output image
*
*********************************************************************************************************
*/

cv::Mat apply_threshold(cv::Mat const& sob_x, cv::Mat const& sob_y, cv::Mat const& magn, cv::Mat const& dir, std::string file_extension) {

	cv::Mat sob, bin, gradient;

	if (verbose) {
		show_and_save("Thresholded_Gradient_x", sob_x, file_extension, true, false);
		show_and_save("Thresholded_Gradient_y", sob_y, file_extension, true, false);
		show_and_save("Thresholded_Magnitude", magn, file_extension, true, true);
		show_and_save("Thresholded_Direction", dir, file_extension, true, true);
	}

	// Combine thresholded gradient binary images through a bitwise AND operation
	bitwise_and(sob_x, sob_y, sob);  // or sob = sob_x & sob_y;
	
	// Combine thresholded gradient Magnitude binary image with thresholded gradient Direction binary image through a bitwise AND operation
	bitwise_and(magn, dir, bin);

	// Combine together the thresholded binary images through a bitwise OR operation
	bitwise_or(sob, bin, gradient);

	if (verbose) {
		show_and_save("Gradient_Threshold", sob, file_extension, true, true);
		show_and_save("Combined_Threshold", bin, file_extension, true, true);
		show_and_save("Combined_Gradient", gradient, file_extension, true, true);
	}
	return gradient;
}

/*
*********************************************************************************************************
*                                            apply_color_threshold FUNCTION
*
* Description : This function convert an RGB image to 4 color spaces: RGB, HLS, LAB and LUV in order to
				pick up the lane lines, then apply the thresholds on:
				- the R channel of RGB color space,
				- the S channel of HLS color space,
				- the B channel of LAB color space,
				- the L channel of LUV color space,
				Creating binary images with threshold it is possible to correctly isolate the pixels of the lane line
				using appropriate channels (the best ones) of different color spaces.
				Finally, combine them together in a binary output image
*
*
*********************************************************************************************************
*/

void apply_color_threshold(cv::Mat const& img, cv::Mat& color, std::string file_extension) {

	cv::Mat hls, lab, luv; // declaration of color space matrices
	cv::Mat rgb_channels[3], hls_channels[3], lab_channels[3], luv_channels[3]; // declaration of channel matrices of different color spaces
	cv::Mat r_binary, s_binary, b_binary, l_binary; // binary threshold channels
	vector<cv::Mat> rgb_color_spaces, hls_color_spaces, lab_color_spaces, luv_color_spaces; // vector of Mat used to store the color spaces
	vector<cv::Mat> color_threshold; // vector of Mat used to store the binary threshold images
	cv::Mat color_rs, color_rsb; // combined binary images of relevant channels

	// Convert RGB image to HLS color space (Hue Light Saturation)
	cv::cvtColor(img, hls, cv::COLOR_BGR2HLS);
	// Convert RGB image to LAB color space
	cv::cvtColor(img, lab, cv::COLOR_BGR2Lab);
	// Convert RGB image to LUV color space
	cv::cvtColor(img, luv, cv::COLOR_BGR2Luv);

	// Isolate each color channel from each color space and display them

	cv::split(img, rgb_channels);
	cv::split(hls, hls_channels);
	cv::split(lab, lab_channels);
	cv::split(luv, luv_channels);

	// RGB channels
	rgb_color_spaces.push_back(rgb_channels[0]); // R channel
	rgb_color_spaces.push_back(rgb_channels[1]); // G channel
	rgb_color_spaces.push_back(rgb_channels[2]); // B channel

	// display_images(rgb_color_spaces, "RGB channel");

	// HLS channels
	hls_color_spaces.push_back(hls_channels[0]); // H channel
	hls_color_spaces.push_back(hls_channels[1]); // L channel
	hls_color_spaces.push_back(hls_channels[2]); // S channel

	// display_images(hls_color_spaces, "HLS channel");

	// LAB channels
	lab_color_spaces.push_back(lab_channels[0]); // L channel
	lab_color_spaces.push_back(lab_channels[1]); // A channel
	lab_color_spaces.push_back(lab_channels[2]); // B channel

	// display_images(lab_color_spaces, "LAB channel");

	// LUV channels
	luv_color_spaces.push_back(luv_channels[0]); // L channel
	luv_color_spaces.push_back(luv_channels[1]); // U channel
	luv_color_spaces.push_back(luv_channels[2]); // V channel

	// display_images(luv_color_spaces, "LUV channel");

	// Apply a threshold to each relevant channel

	color_thresh(rgb_channels[0], r_binary, 'r'); // R binary threshold
	color_thresh(hls_channels[2], s_binary, 's'); // S binary threshold
	color_thresh(lab_channels[2], b_binary, 'b'); // B binary threshold
	color_thresh(luv_channels[0], l_binary, 'l'); // L binary threshold

	// display them
	color_threshold.push_back(r_binary);
	color_threshold.push_back(s_binary);
	color_threshold.push_back(b_binary);
	color_threshold.push_back(l_binary);

	if (verbose) {
		display_images(color_threshold, "Color");
	}
	// Bitwise OR operation to sum different thresholded color binary images 

	cv::bitwise_or(r_binary, s_binary, color_rs); // or color_rs = r_binary | s_binary;
	cv::bitwise_or(color_rs, b_binary, color_rsb);
	cv::bitwise_or(color_rsb, l_binary, color);

	if (verbose) {
		show_and_save("Thresholded_Color", color, file_extension, true, true);
	}
	return;
}

/*
*********************************************************************************************************
*                                            color_thresh FUNCTION
*
* Description : This function is used to correctly filter each color channel (R channel from RGB color space,
				S channel from HLS color space, B channel from LAB color space, L channel from LUV
				color space) returning a thresholded binary image
*
*
*********************************************************************************************************
*/

void color_thresh(cv::Mat const& channels, cv::Mat& binary, char type_color) {

	// Range based thresholding:
	// pixels between color_channel_Threshold_min and color_channel_Threshold_max set to 1, otherwise 0

	int r_thresh_min = 145;
	int r_thresh_max = 255;
	int s_thresh_min = 80;
	int s_thresh_max = 255;
	int b_thresh_min = 150;
	int b_thresh_max = 200;
	int l_thresh_min = 145;
	int l_thresh_max = 255;

	if (type_color == 'r') {
		cv::inRange(channels, cv::Scalar(r_thresh_min), cv::Scalar(r_thresh_max), binary);
	}

	if (type_color == 's') {
		cv::inRange(channels, cv::Scalar(s_thresh_min), cv::Scalar(s_thresh_max), binary);
	}

	if (type_color == 'b') {
		cv::inRange(channels, cv::Scalar(b_thresh_min), cv::Scalar(b_thresh_max), binary);
	}

	if (type_color == 'l') {
		cv::inRange(channels, cv::Scalar(l_thresh_min), cv::Scalar(l_thresh_max), binary);
	}

	return;
}

/*
*********************************************************************************************************
*                                            apply_combined_threshold FUNCTION
*
* Description : This function is used to combine together the thresholded binary gradient and the thresholded binary color
				in one single binary output image where it is possible to adequately distinguish the lane lines
*
*********************************************************************************************************
*/

cv::Mat apply_combined_threshold(cv::Mat const& gradient, cv::Mat const& color, std::string file_extension) {

	cv::Mat dest;

	// Bitwise OR operation to sum thresholded binary gradient and thresholded binary color images
	bitwise_or(gradient, color, dest); // or dst = color | gradient;
	if (verbose) {
		show_and_save("Combined_final", dest, file_extension, true, true);
	}
	return dest;
}

/*
*********************************************************************************************************
*                                            display_images FUNCTION
*
* Description : This function is used to display multiple images on the screen
*
*
*********************************************************************************************************
*/

void display_images(vector<Mat> images, std::string title)
{
	// iterate through the images
	for (int index = 0; index < images.size(); ++index)
	{
		// index allows us to assign a unique name to each window
		// windows will be titled 'Window 0', 'Window 1', 'Window 2'... until frames.size()-1
		cv::String windowTitle = title + " " + to_string(index);

		imshow(windowTitle, images[index]);
	}
	cv::waitKey(0); // wait for any key press
	cv::destroyAllWindows();
}

/*
*********************************************************************************************************
*                                            combined_threshold FUNCTION
*
* Description : This function combines the color thresholds with the gradient thresholds of a RGB input image
				returning as output a thresholded binary image.
				Initially, the Sobel operator is applied to a gray scale image to get the gradients in the x and y direction
				which are also used to obtain the gradient magnitude and direction thresholded images combining everything into
				a single filtered binary image.
				Then, Thresholded binary image, combining different color channels of different color spaces,
				is created from the original RBG image.
				Finally, Thresholded Binary Gradient and Thresholded Binary Color are combined together in order to obtain
				the best binary image for identifying lane lines.
*
*
*********************************************************************************************************
*/

void combined_threshold(cv::Mat const& img, cv::Mat& dst, std::string extension)
{
	cv::Mat gray, sobel_x, sobel_y, magn, dir, gradient, color;

	// Convert to gray -> A gray-scale image (black and white image) only consists of a single channel.
	// So the gray image does not contain any color information.
	// The higher the pixel value, the brighter it is and vice-versa.

	cv::cvtColor(img, gray, cv::COLOR_BGR2GRAY);

	// Apply Sobel thresholding in parallel
	std::thread x_direct(abs_sobel_threshold, std::ref(gray), std::ref(sobel_x), 'x', 3, 20, 255);
	std::thread y_direct(abs_sobel_threshold, std::ref(gray), std::ref(sobel_y), 'y', 3, 20, 255);

	// The function returns when the thread execution has completed its task
	x_direct.join();
	y_direct.join();

	// Apply gradient Magnitude threshold
	magnitude_threshold(gray, magn, 3, 20, 100);

	// Apply gradient Direction threshold
	direction_threshold(gray, dir, 3, 0.7, 1.3);

	// Combines Gradient Thresholds
	gradient = apply_threshold(sobel_x, sobel_y, magn, dir, extension);

	// Apply Color Channel Thresholds
	apply_color_threshold(img, color, extension);

	// Combines the thresholded binary gradient with the thresholded binary color
	dst = apply_combined_threshold(gradient, color, extension);

	return;
}

/*
************************************************************************************************************************
*                                              ROI FUNCTION
*
* Description: This function is used to show the Region of Interest on an image
*
*
************************************************************************************************************************
*/

void ROI(const cv::Mat& image, std::vector<cv::Point2f> const& points, std::string extension)
{
	// Compute the trapezoid vertices using the source points -> Draws lines segment connecting the 4 points.
	// For the second call: Compute the vertices of the rectangle using the destination points
	cv::line(image, points[0], points[1], Scalar(255), 2, 8, 0);
	cv::line(image, points[1], points[2], Scalar(255), 2, 8, 0);
	cv::line(image, points[2], points[3], Scalar(255), 2, 8, 0);
	cv::line(image, points[3], points[0], Scalar(255), 2, 8, 0);

	// Show the ROI
	if (verbose) {
		show_and_save("ROI", image, extension, true, true);
	}
	return;
}

/*
************************************************************************************************************************
*                                              binary_transform FUNCTION
*
* Description: This function is used to transform the input thresholded binary image into a binary warped image.
			   So, a perspective transformation is applied in order to obtain a birds-eye view of the lane.
			   To do this, 2 sets of points are used to also calculating Transformation Matrix: 4 coordinates of source points
			   to set the ROI in the original image and 4 coordinates of destination points to map them in the warped image.
*
*
************************************************************************************************************************
*/

void binary_transform(const cv::Mat& image, cv::Mat& warped, cv::Mat& M, cv::Mat& Minv, cv::Mat& combined, std::string extension)
{
	std::vector<cv::Point2f> src, dst; // used to store the source and destination points
	float width, height;
	cv::Mat original_warp;
	cv::Size siz = image.size();
	height = (float)siz.height; // height 720
	width = (float)siz.width; // width 1280

	// Search for source points in order to map them into different image destination points with a new perspective 
	// calculate the source vertices of the Region of Interest from the input image
	cv::Point2f left_top, left_bottom, right_top, right_bottom;

	left_top = Point2f(605.f, 395);
	right_top = Point2f(710.f, 395);
	right_bottom = Point2f(1037.f, height);
	left_bottom = Point2f(255.f, height);

	src.push_back(left_bottom);
	src.push_back(left_top);
	src.push_back(right_top);
	src.push_back(right_bottom);

	// calculate the destination points of the warp image (destination points are vertices of output image)
	dst.push_back(cv::Point2f(250, height));
	dst.push_back(cv::Point2f(250, 0));
	dst.push_back(cv::Point2f(900, 0));
	dst.push_back(cv::Point2f(900, height));

	// Show the ROI
	ROI(image.clone(), src, extension);

	// given the source and destination points calculate the perspective transformation matrix
	// and its inverse for perspective warp

	// Get Transformation Matrix -> used for Bird's eye view
	M = cv::getPerspectiveTransform(src, dst);

	// Get Inverse Transformation Matrix -> used to return to the normal view
	Minv = cv::getPerspectiveTransform(dst, src);

	// Applies a perspective transformation to the image to get a bird's eye view image
	cv::warpPerspective(image, original_warp, M, image.size(), cv::INTER_LINEAR);

	// Applies a perspective transformation to the binary image to get a bird's eye view binary image
	cv::warpPerspective(combined, warped, M, combined.size(), cv::INTER_LINEAR); // binary birds-eye view image
	if (verbose) {
		show_and_save("Warped_Image", original_warp, extension, true, false);
		show_and_save("Binary_Warped_Image", warped, extension, true, true);
	}

	ROI(original_warp.clone(), dst, extension);

	return;
}

/*
************************************************************************************************************************
*                                              polyfit FUNCTION
*
* Description: This function is used to perform a polynomial fitting
*
*
************************************************************************************************************************
*/

void polyfit(const cv::Mat& src_x, const cv::Mat& src_y, cv::Mat& dst, int order)
{
	// CV_Assert Checks a condition at runtime and throws exception if it fails
	CV_Assert((src_x.rows > 0) && (src_y.rows > 0) && (src_x.cols == 1) && (src_y.cols == 1)
		&& (dst.cols == 1) && (dst.rows == (order + 1)) && (order >= 1));

	cv::Mat X;
	X = cv::Mat::zeros(src_x.rows, order + 1, CV_32FC1);
	cv::Mat copy;

	for (int i = 0; i <= order; i++)
	{
		copy = src_x.clone();
		pow(copy, i, copy);
		cv::Mat M1 = X.col(i);
		copy.col(0).copyTo(M1);
	}
	cv::Mat X_t, X_inv;
	transpose(X, X_t);
	cv::Mat temp = X_t * X;
	cv::Mat temp2;
	invert(temp, temp2);
	cv::Mat temp3 = temp2 * X_t;
	cv::Mat W = temp3 * src_y;
	W.copyTo(dst);
}